<?php
session_start();
require_once 'classes/User.php';
require_once 'classes/Security.php';

$security = new Security();

if(!isset($_GET['token'])) {
    header("Location: login.php");
    exit();
}

$user = new User();
$token = $_GET['token'];

if(!$user->validateResetToken($token)) {
    $_SESSION['error'] = "Invalid or expired password reset link.";
    header("Location: login.php");
    exit();
}

if(isset($_POST['reset_password'])) {
    // CSRF Protection
    if(!$security->validateCSRFToken($_POST['csrf_token'])) {
        $error = "Invalid request. Please try again.";
    } else {
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];
        
        if($new_password !== $confirm_password) {
            $error = "Passwords do not match.";
        } else {
            // Password strength validation
            $password_errors = $security->validatePasswordStrength($new_password);
            if(!empty($password_errors)) {
                $error = "Password does not meet the requirements:<br>" . implode("<br>", $password_errors);
            } else {
                // Get user ID from token
                $user_id = $user->getUserIdFromToken($token);
                
                // Check password history
                if(!$security->checkPasswordHistory($user_id, $new_password)) {
                    $error = "You cannot reuse any of your last " . Security::PASSWORD_HISTORY_LIMIT . " passwords.";
                } else {
                    if($user->resetPassword($token, $new_password)) {
                        // Add to password history
                        $security->addToPasswordHistory($user_id, $new_password);
                        
                        $_SESSION['success'] = "Password has been reset successfully. Please login with your new password.";
                        header("Location: login.php");
                        exit();
                    } else {
                        $error = "Failed to reset password. Please try again.";
                    }
                }
            }
        }
    }
}

$csrf_token = $security->generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h3 class="text-center">Reset Password</h3>
                    </div>
                    <div class="card-body">
                        <?php if(isset($error)): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>
                        
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            <div class="mb-3">
                                <label for="new_password" class="form-label">New Password</label>
                                <input type="password" class="form-control" id="new_password" name="new_password" required>
                                <div class="form-text">
                                    Password must:
                                    <ul>
                                        <li>Be at least <?php echo MIN_PASSWORD_LENGTH; ?> characters long</li>
                                        <?php if(REQUIRE_UPPERCASE): ?>
                                            <li>Contain at least one uppercase letter</li>
                                        <?php endif; ?>
                                        <?php if(REQUIRE_LOWERCASE): ?>
                                            <li>Contain at least one lowercase letter</li>
                                        <?php endif; ?>
                                        <?php if(REQUIRE_NUMBERS): ?>
                                            <li>Contain at least one number</li>
                                        <?php endif; ?>
                                        <?php if(REQUIRE_SPECIAL_CHARS): ?>
                                            <li>Contain at least one special character</li>
                                        <?php endif; ?>
                                        <li>Not be one of your last <?php echo Security::PASSWORD_HISTORY_LIMIT; ?> passwords</li>
                                        <li>Not be a common password</li>
                                    </ul>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label for="confirm_password" class="form-label">Confirm New Password</label>
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                            </div>
                            <div class="d-grid">
                                <button type="submit" name="reset_password" class="btn btn-primary">Reset Password</button>
                            </div>
                        </form>
                        <div class="text-center mt-3">
                            <a href="login.php">Back to Login</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html> 