<?php
require_once __DIR__ . '/../config/mail.php';
require_once __DIR__ . '/../config/database.php';

class Security {
    private $conn;
    private const PASSWORD_HISTORY_LIMIT = 5; // Number of previous passwords to check
    private const PASSWORD_EXPIRY_DAYS = 90; // Password expires after 90 days
    
    public function __construct() {
        $this->conn = getDBConnection();
        // Clean up expired data on initialization
        $this->cleanupExpiredData();
    }
    
    // CSRF Protection
    public function generateCSRFToken() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
    
    public function validateCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
            return false;
        }
        return true;
    }
    
    // Rate Limiting
    public function checkRateLimit($email) {
        $query = "SELECT COUNT(*) as count FROM password_reset_attempts 
                 WHERE email = :email 
                 AND attempt_time > DATE_SUB(NOW(), INTERVAL :window SECOND)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":email", $email);
        $stmt->bindParam(":window", RESET_REQUEST_WINDOW);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['count'] >= RESET_REQUEST_LIMIT) {
            return false;
        }
        
        // Log this attempt
        $this->logResetAttempt($email);
        return true;
    }
    
    private function logResetAttempt($email) {
        $query = "INSERT INTO password_reset_attempts (email, attempt_time) VALUES (:email, NOW())";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":email", $email);
        $stmt->execute();
    }
    
    // Password History Management
    public function checkPasswordHistory($user_id, $new_password) {
        $query = "SELECT password FROM password_history 
                 WHERE user_id = :user_id 
                 ORDER BY created_at DESC 
                 LIMIT " . self::PASSWORD_HISTORY_LIMIT;
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->execute();
        
        while($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            if(password_verify($new_password, $row['password'])) {
                return false; // Password found in history
            }
        }
        return true;
    }
    
    public function addToPasswordHistory($user_id, $password) {
        $query = "INSERT INTO password_history (user_id, password, created_at) 
                 VALUES (:user_id, :password, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":password", $password);
        $stmt->execute();
        
        // Update user's last password change
        $query = "UPDATE users 
                 SET last_password_change = NOW(),
                     password_expiry_date = DATE_ADD(NOW(), INTERVAL " . self::PASSWORD_EXPIRY_DAYS . " DAY)
                 WHERE id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->execute();
    }
    
    public function checkPasswordExpiry($user_id) {
        $query = "SELECT password_expiry_date FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if($result && $result['password_expiry_date']) {
            return strtotime($result['password_expiry_date']) <= time();
        }
        return false;
    }
    
    public function getDaysUntilPasswordExpiry($user_id) {
        $query = "SELECT password_expiry_date FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if($result && $result['password_expiry_date']) {
            $expiry = strtotime($result['password_expiry_date']);
            $now = time();
            return max(0, ceil(($expiry - $now) / (60 * 60 * 24)));
        }
        return null;
    }

    // Enhanced Password Strength Validation
    public function validatePasswordStrength($password) {
        $errors = [];
        
        if (strlen($password) < MIN_PASSWORD_LENGTH) {
            $errors[] = "Password must be at least " . MIN_PASSWORD_LENGTH . " characters long";
        }
        
        if (REQUIRE_UPPERCASE && !preg_match('/[A-Z]/', $password)) {
            $errors[] = "Password must contain at least one uppercase letter";
        }
        
        if (REQUIRE_LOWERCASE && !preg_match('/[a-z]/', $password)) {
            $errors[] = "Password must contain at least one lowercase letter";
        }
        
        if (REQUIRE_NUMBERS && !preg_match('/[0-9]/', $password)) {
            $errors[] = "Password must contain at least one number";
        }
        
        if (REQUIRE_SPECIAL_CHARS && !preg_match('/[!@#$%^&*()\-_=+{};:,<.>]/', $password)) {
            $errors[] = "Password must contain at least one special character";
        }
        
        // Check for common passwords
        if($this->isCommonPassword($password)) {
            $errors[] = "This password is too common. Please choose a stronger password.";
        }
        
        return $errors;
    }
    
    private function isCommonPassword($password) {
        // List of common passwords (you should expand this list)
        $common_passwords = [
            'password123', '12345678', 'qwerty123', 'admin123',
            'welcome123', 'letmein123', 'password1', '123456789'
        ];
        
        return in_array(strtolower($password), $common_passwords);
    }
    
    // Cleanup expired data
    private function cleanupExpiredData() {
        // Clean up expired reset tokens
        $query = "UPDATE users 
                 SET reset_token = NULL, 
                     reset_token_expiry = NULL 
                 WHERE reset_token_expiry < NOW()";
        $this->conn->exec($query);
        
        // Clean up old reset attempts (older than 24 hours)
        $query = "DELETE FROM password_reset_attempts 
                 WHERE attempt_time < DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        $this->conn->exec($query);
    }

    // Add IP-based rate limiting
    public function checkIPRateLimit($ip) {
        $query = "SELECT COUNT(*) as count FROM password_reset_attempts 
                 WHERE ip_address = :ip 
                 AND attempt_time > DATE_SUB(NOW(), INTERVAL :window SECOND)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":ip", $ip);
        $stmt->bindParam(":window", RESET_REQUEST_WINDOW);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['count'] >= RESET_REQUEST_LIMIT) {
            return false;
        }
        
        return true;
    }

    // Enhanced logging for security events
    private function logSecurityEvent($event_type, $user_id = null, $ip_address = null) {
        $query = "INSERT INTO security_logs 
                 (event_type, user_id, ip_address, event_time) 
                 VALUES (:event_type, :user_id, :ip_address, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":event_type", $event_type);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":ip_address", $ip_address);
        $stmt->execute();
    }

    // Enhanced email sending with retry mechanism
    public function sendPasswordResetEmail($to, $resetLink) {
        require 'vendor/autoload.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        $max_retries = 3;
        $retry_count = 0;
        
        while ($retry_count < $max_retries) {
            try {
                // Server settings
                $mail->isSMTP();
                $mail->Host = SMTP_HOST;
                $mail->SMTPAuth = true;
                $mail->Username = SMTP_USERNAME;
                $mail->Password = SMTP_PASSWORD;
                $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
                $mail->Port = SMTP_PORT;
                
                // Add timeout settings
                $mail->Timeout = 30;
                $mail->SMTPKeepAlive = true;
                
                // Recipients
                $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
                $mail->addAddress($to);
                
                // Content
                $mail->isHTML(true);
                $mail->Subject = 'Password Reset Request';
                $mail->Body = "
                    <h2>Password Reset Request</h2>
                    <p>You have requested to reset your password. Click the link below to proceed:</p>
                    <p><a href='{$resetLink}'>{$resetLink}</a></p>
                    <p>This link will expire in 1 hour.</p>
                    <p>If you did not request this password reset, please ignore this email.</p>
                    <p>For security reasons, this link can only be used once.</p>
                ";
                
                $mail->send();
                $this->logSecurityEvent('password_reset_email_sent', null, $_SERVER['REMOTE_ADDR']);
                return true;
            } catch (Exception $e) {
                $retry_count++;
                error_log("Email sending failed (attempt {$retry_count}): " . $mail->ErrorInfo);
                if ($retry_count < $max_retries) {
                    sleep(2); // Wait 2 seconds before retrying
                }
            }
        }
        
        $this->logSecurityEvent('password_reset_email_failed', null, $_SERVER['REMOTE_ADDR']);
        return false;
    }
}
?> 